﻿/*
VERSION: 1.0

USAGE:
	#include "functions/walk_tank.as"
	player_mc.walk_obj = makeWalkTank( player_mc );
	
	onUnload = function(){
		clearInterval(walk_obj.loopInterval);
	}
	
	
SETTINGS:
	walkSpeed			Maximum speed  (in pixels per frame)
	acel					Acelleration speed,  how many pixels the velocity increases per frame while a key is pressed
	decel					Deceleration speed, multiplier that reduces velocity per frame when no key is pressed
	turn_maxVel		Maximum rotation speed  (in radians)
	turn_acel			Rotation acelleration,  how many radians to rotate per frame while a key is pressed
	turn_decel		Rotation deceleration, multiplier that reduces velocity per frame when no key is pressed
	
	key_moveForward		Key code for moving forward
	key_moveBackward	Key code for moving backward
	key_turnLeft			Key code for rotating left
	key_turnRight			Key code for rotating right
	
	
PROPERTIES:
	xVel				Current horizontal speed  (in pixels per frame)
	yVel				Current vertical speed  (in pixels per frame)
	
NOTES:
	This system has optional hooks for these collision and sprite systems:
		sprite.as
		WalkCollision.as
	It will use them if they're present.
	
ADDING COLLISION:
	A collision system can be added like this:
		walk_obj.collision = new Collision( player_mc, walk_obj, collision_array, -8, -8 );
	The walk_obj is this walk system.

*/
function makeWalkTank( player_mc, playerSprite_mc ){
	#include "eventSystem3.as"
	var _this = {};		// external interface
	AsBroadcaster.initialize( _this );
	var react = make_react( _this );		// param is optional
	react.to("unload").from( player_mc ).tell( _this );
	
	
	
	// Variables
	var fps = 30;
	// movement speed
	_this.turn_acel = Math.PI /180 *3;				// acelleration increment		(3 degrees of acelleration)
	_this.turn_decel = 0.6;				// brake multiplier
	_this.turn_vel = 0;
	_this.turn_maxVel = Math.PI /180 *14;		// 14 degrees of rotation
	_this.turn_radian = Math.PI /180 *90;		// current rotation  (in radians)
	_this.move_acel = 1;				// acelleration increment
	_this.move_decel = 0.6;			// brake multiplier
	_this.move_speed = 0;				// current movement speed
	_this.move_maxSpeed = 10;
	// cartesian velocities for:   external collision systems  =>  final movement
	_this.xVel = 0;
	_this.yVel = 0;
	// controls
	_this.key_turnLeft = RAM.keys.left || Key.LEFT;
	_this.key_turnRight = RAM.keys.right || Key.RIGHT;
	_this.key_moveForward = RAM.keys.up || Key.UP;
	_this.key_moveBackward = RAM.keys.down || Key.DOWN;
	// allow the player's image-sprite to be swapped-out later
	_this.playerSprite_mc = playerSprite_mc;
	// allow the "radian" rotation to be accessed externally
	_this.addProperty( "radian", function(){
		return _this.turn_radian
	}, function( newValue ){
		_this.turn_radian = newValue;
	});
	
	// legacy
	// // move speed
	_this.addProperty( "walkSpeed", function(){
		return _this.move_maxSpeed;
	}, function( newValue ){
		_this.move_maxSpeed = newValue;
	});
	_this.addProperty( "acel", function(){
		return _this.move_acel
	}, function( newValue ){
		_this.move_acel = newValue;
	});
	_this.addProperty( "decel", function(){
		return _this.move_decel
	}, function( newValue ){
		_this.move_decel = newValue;
	});
	// // controls
	_this.addProperty( "up_key", function(){
		return _this.key_moveForward
	}, function( newValue ){
		_this.key_moveForward = newValue;
	});
	_this.addProperty( "down_key", function(){
		return _this.key_moveBackward
	}, function( newValue ){
		_this.key_moveBackward = newValue;
	});
	_this.addProperty( "left_key", function(){
		return _this.key_turnLeft
	}, function( newValue ){
		_this.key_turnLeft = newValue;
	});
	_this.addProperty( "right_key", function(){
		return _this.key_turnRight
	}, function( newValue ){
		_this.key_turnRight = newValue;
	});
	
	// Data request events
	react.to("request_radian").then = function( evt ){
		evt.callback( _this.turn_radian );
	}
	react.to("request_radian").from( player_mc ).then = function( evt ){
		evt.callback( _this.turn_radian );
	}
	
	
	
	
	
	// // // // // // // // // // // // // // // // // // // // // // // // // // // // // // 
	// Main Program
	_this.loop = function(){
		//  Abort if the player sprite doesn't exist
		if(player_mc._currentframe === undefined){
			sendEvent("unload");
			return;
		}// if:   player_mc is missing
		
		_this.turn_vel = changeTurnVelocity( _this.turn_vel );
		// update total speeds
		_this.move_speed = changeMoveVelocity( _this.move_speed );		// pixel movement magnitude
		_this.turn_radian += _this.turn_vel;		// radian
		_this.move_speed = limitMoveSpeed( _this.move_speed );
		
		// announce rotation
		if(_this.turn_vel !== 0){
			sendEvent( "change_radian", {value: _this.turn_radian}, _this);
			sendEvent( "change_radian", {value: _this.turn_radian}, player_mc);
		}
		// get cartesian velocities
		var velocity_vec = get_velocity_vec( _this.move_speed, _this.turn_radian );
		_this.xVel = velocity_vec.x;
		_this.yVel = velocity_vec.y;
		
		_this.collision.run();		// apply external collision  (zeros xVel/yVel when collisions occur)
		setAnimSpeed( _this.xVel, _this.yVel );
		applyMovement( _this.xVel, _this.yVel );
	}// loop()
	_this.loopInterval = setInterval( function(){
		_this.loop();
	}, 1000/fps );
	react.to("unload").then = function(){
		clearInterval(_this.loopInterval);
	}
	
	return _this;
	
	
	
	
	
	// // // // // // // // // // // // // // // // // // // // // // // // // // // // // // 
	function limitMoveSpeed( move_speed ){
		if( move_speed > _this.move_maxSpeed )		move_speed = _this.move_maxSpeed;
		if( move_speed < -_this.move_maxSpeed )		move_speed = -_this.move_maxSpeed;
		return move_speed;
	}// limitMoveSpeed()
	
	
	
	function decellerateMoveSpeed( move_speed ){
		move_speed *= _this.move_decel;
		if( Math.abs( move_speed ) < 1 )		move_speed = 0;		// if less than 1 pixel of movement,  then => no movement
		return move_speed;
	}// decellerateMoveSpeed()
	
	
	
	function applyMovement( x, y ){
		player_mc._x += x;
		player_mc._y += y;
	}// applyMovement()
	
	
	
	function setAnimSpeed( xVel, yVel ){
		var xVel = Math.abs(xVel);
		var yVel = Math.abs(yVel);
		// adjust walk speed
		var max = 5;
		var range = 4;
		var topSpeed = 10;		// used for animation
		//var moveSpeed = Math.sqrt(_this.xVel*_this.xVel + _this.yVel*_this.yVel);
		var moveSpeed =  (xVel > yVel)  ?  xVel  :  yVel;		// Use the fastest velocity
		var speedPercent = Math.abs(moveSpeed) / topSpeed;		// get percent
		var animSpeed = max - (range * speedPercent);
		var animSpeed = (animSpeed<1) ? 1 : animSpeed;		// enforce max speed
		_this.playerSprite_mc.setParams({delay:animSpeed});
	}// setAnimSpeed()
	
	
	
	function get_velocity_vec( moveSpeed, turnSpeed ){
		var Point = flash.geom.Point;
		var velocity_vec = Point.polar( -moveSpeed, turnSpeed );
		return velocity_vec;
	}// get_velocity_vec()
	
	
	
	function changeTurnVelocity( turn_vel ){
		// turn
		if( _this.key_turnRight  &&  Key.isDown( _this.key_turnRight ) ){
			if( turn_vel < _this.turn_maxVel ){
				if(turn_vel < 0)		decellerateTurn();
				turn_vel += _this.turn_acel;
				// limit the max speed
				if( turn_vel > _this.turn_maxVel )		turn_vel = _this.turn_maxVel;
			}// if:   not at or beyond the max speed
		}else if( _this.key_turnLeft  &&  Key.isDown( _this.key_turnLeft ) ){
			if( turn_vel > -_this.turn_maxVel ){
				if(turn_vel > 0)		decellerateTurn();
				turn_vel -= _this.turn_acel;
				// limit the max speed
				if( turn_vel < -_this.turn_maxVel )		turn_vel = -_this.turn_maxVel;
			}// if:   not at or beyond the max speed
		}else{
			decellerateTurn();
		}
		function decellerateTurn(){
			turn_vel *= _this.turn_decel;
			if( Math.abs( turn_vel ) < 0.017 )		turn_vel = 0;		// if less than 1 degree of rotation,  then => no rotation
		}// decellerateTurn()
		
		return turn_vel;
	}// changeTurnVelocity()
	
	
	
	
	function changeMoveVelocity( move_speed ){
		// move
		if( _this.key_moveForward  &&  Key.isDown( _this.key_moveForward ) ){
			if( move_speed < _this.move_maxSpeed ){
				if(move_speed < 0)		decellerateMove();
				move_speed += _this.move_acel;
			}// if:   not at or beyond the max speed
		}else if( _this.key_moveBackward  &&  Key.isDown( _this.key_moveBackward ) ){
			if( move_speed > -_this.move_maxSpeed ){
				if(turn_vel > 0)		decellerateMove();
				move_speed -= _this.move_acel;
			}// if:   not at or beyond the max speed
		}else{
			decellerateMove();
		}
		function decellerateMove(){
			move_speed *= _this.move_decel;
			if( Math.abs( move_speed ) < 1 )		move_speed = 0;		// if less than 1 pixel of movement,  then => no movement
		}// decellerateMove()
		
		if( move_speed > _this.move_maxSpeed )		move_speed = _this.move_maxSpeed;
		if( move_speed < -_this.move_maxSpeed )		move_speed = -_this.move_maxSpeed;
		
		return move_speed;
	}// changeMoveVelocity()
	
}// makeWalkTank()